/* $XConsortium: mxic_driver.c,v 1.4 95/01/16 13:18:25 kaleb Exp $ */
/* $XFree86: xc/programs/Xserver/hw/xfree86/vga256/drivers/tvga8900/mxic_driver.c,v 3.9 1995/05/27 03:17:34 dawes Exp $ */
/*
 * Copyright 1992 by Alan Hourihane, Wigan, England.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Alan Hourihane not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  Alan Hourihane makes no representations
 * about the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 *
 * ALAN HOURIHANE DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL ALAN HOURIHANE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Stephen H.L.Wang
 *
 * Description:
 * 	This porting support only MX86250 and MX86251 because the extended
 *      register set and the programming procedure of these two chipsets
 * 	are similiar.  These two chipsets use the same VCG and they use
 *	programmable clock generator.
 *
 * version 0.2.1:
 * 11/8/1998
 *	1. Fix RAMDAC mode setting problem in high/true color mode.
 *	   High/true color can work now.
 *	2. Enable H/W cursor. 
 *	3. Update some indexed clock value to support more modes.
 *
 *	TODO in next revision:
 * 	1. Mode setting has CRTC FIFO setting bug in MX86250 high/true color.
 *	2. Add programmable clock support.
 *
 * version 0.2alpha:
 *      1. Add H/W acceleration functions.  Currently only ScreenToScreenCopy and
 *      SolidFill were implemented.
 *      2. Add 2048 pixels width support.
 *      3. Add PCI probing.
 *
 * version 0.1alpha:
 *	1. Use predefined values of programmable VCG to simluate indexed 
 *	clock selection.
 */

#include "X.h"
#include "input.h"
#include "screenint.h"
#include "dix.h"

#include "compiler.h"

#include "xf86.h"
#include "xf86Priv.h"
#include "xf86_OSlib.h"
#include "xf86_HWlib.h"
#define XCONFIG_FLAGS_ONLY
#include "xf86_Config.h"
#include "vga.h"

/* DGA includes */
#ifdef XFreeXDGA
#include "X.h"
#include "Xproto.h"
#include "scrnintstr.h"
#include "servermd.h"
#define _XF86DGA_SERVER_
#include "extensions/xf86dgastr.h"
#endif

#include "mxic_driver.h"

#ifdef FORCEMODEREG
#include "mxic_mode.h"
#endif

#ifdef XF86VGA16
#define MONOVGA
#endif

typedef struct {
	vgaHWRec std;          		/* std IBM VGA register 	*/
	/* 
	   The following fields before union field are common to
	   MX86250 and MX86251
	*/
	unsigned char CRTC21;
	unsigned char CRTC22;
	unsigned char CRTC23;
 	unsigned char CRTC24;
 	unsigned char CRTC66;
	unsigned char SEQ11;
	unsigned char SEQ18;
	unsigned char SEQ19;
	unsigned char SEQ1D;
	unsigned char SystemControl;	
	/*
	   The following fields are different between 
	   MX86250 and MX86251
	*/
	union {
		struct {
			unsigned char CRTC33;
			unsigned char CRTC34;
			unsigned char CRTC35;
			unsigned char CRTC36;
			unsigned char CRTC37;
			unsigned char CRTC38;
			unsigned char CRTC39;
			unsigned char CRTC3A;
			unsigned char CRTC3B;
			unsigned char CRTC3C;
			unsigned char CRTC3D;
			unsigned char CRTC3E;
			unsigned char CRTC3F;
		} MX86250;
		struct {
			unsigned char CRTC1F;
			unsigned char CRTC20;
		} MX86251;
	} extreg;
#ifdef FORCEMODEREG
	unsigned char CRTCALL[0x3f];
	unsigned char SEQALL[0x1f];
#endif
} vgaMXICRec, *vgaMXICPtr;

static Bool MXICClockSelect();
static char *MXICIdent();
static Bool MXICProbe();
static void MXICEnterLeave();
static Bool MXICInit();
static Bool MXICValidMode();
static void *MXICSave();
static void MXICRestore();
static void MXICAdjust();
static void MXICFbInit();
static Bool MXICForceRestore();

/* external routines defined in mxic_bank.s */
extern void MXICSetRead86250();
extern void MXICSetWrite86250();
extern void MXICSetReadWrite86250();
extern void MXICSetRead86251();
extern void MXICSetWrite86251();
extern void MXICSetReadWrite86251();

extern vgaHWCursorRec vgaHWCursor;

vgaVideoChipRec MXIC = {
  MXICProbe,
  MXICIdent,
  MXICEnterLeave,
  MXICInit,
  MXICValidMode,
  MXICSave,
  MXICRestore,
  MXICAdjust,
  vgaHWSaveScreen,
  (void (*)())NoopDDA,
  MXICFbInit,
  MXICSetRead86250,
  MXICSetWrite86250,
  MXICSetReadWrite86250,
  0x10000,			/* ChipMapSize */
  0x10000,			/* ChipSegmentSize, 16k*/
  16,				/* ChipSegmentShift */
  0xffff,			/* ChipSegmentMask */
  0x00000, 0x10000,		/* ChipReadBottom, ChipReadTop  */
  0x00000, 0x10000,		/* ChipWriteBottom,ChipWriteTop */
  FALSE,                        /* Uses a single bank */
  VGA_DIVIDE_VERT,		/* ChipInterlaceType -- divide verts */
  {0,},				/* ChipOptionFlags */
  8,				/* ChipRounding: Set to 16 in Probe() */
  TRUE,			        /* ChipUseLinearAddressing */
  0,				/* ChipLinearBase */
  0,				/* ChipLinearSize */
  TRUE, 			/* ChipHas16bpp */
  TRUE, 			/* ChipHas24bpp */
  TRUE, 			/* ChipHas32bpp */
  NULL,				/* ChipBuiltinModes */
  1,				/* ChipClockScaleFactor */
  1                             /* ChipClockDivFactor */
};

#define new ((vgaMXICPtr)vgaNewVideoState)

/* entries must be in sequence with chipset numbers !! */
SymTabRec mxicChipTable[] = {
   { MXIC_MX86250,  "MX86250"}, 
   { MXIC_MX86251,  "MX86251"},
   { MXIC_UNKNOWN,  "Unknown"},
   { -1,           ""},
   };

static int MXICchipset;

int vgaCRIndex, vgaCRReg;
MXICPRIV mxicPriv;
unsigned char SystemControlRegIdx;
unsigned char BankRegIdx;
MXIC_MMIO_MEM *mxicMmioMem = NULL;

DisplayModePtr extmode = NULL;

typedef struct {
	unsigned char 	SEQ18;
	unsigned char	SEQ19;
} DCLKMapping;

static DCLKMapping dclk[16] = 
{	
	/*{0xe1, 0x06},*/ /*  25MHz - 0 */
	{0x76, 0x4f}, /*  25MHz - 0 */
	{0xf6, 0x59}, /*  28MHz - 1 */
	{0xf0, 0xde}, /*  40MHz - 2 */
	/*{0xf0, 0xde},*/ /*  40MHz - 3 */
	{0x1f, 0x5e}, /*  85MHz - 3 */
	{0xa1, 0x06}, /*  50MHz - 4 */
	{0x1c, 0x4d}, /*  77MHz - 5 */
	{0x3e, 0x4d}, /*  36MHz - 6 */
	{0x26, 0x15}, /*  45MHz - 7 */
	{0x8c, 0x3a}, /* 130MHz - 8 */
	{0x94, 0x57}, /* 120MHz - 9 */
	{0xb0, 0xde}, /*  80MHz -10 */
	{0xe2, 0x0c}, /*  31MHz -11 */
	{0x92, 0x48}, /* 110MHz -12 */
	{0xac, 0x3a}, /*  65MHz -13 */
	{0x14, 0x36}, /*  75MHz -14 */
	{0x1f, 0x68}  /*  94MHz -15 */
};

unsigned char GetCRTC(unsigned char index)
{
  	outb(vgaCRIndex, index);
	return inb(vgaCRReg);
}

unsigned char GetSEQ(unsigned char index)
{
  	outb(0x3c4, index);
	return inb(0x3c5);
}

/* 
 * MXICIdent --
 */
static char *
MXICIdent(n)
	int n;
{
	static char *chipsets[] = {"MX86250", "MX86251"};

	if (n + 1 > sizeof(chipsets) / sizeof(char *))
		return(NULL);
	else
		return(chipsets[n]);
}

#ifdef DEBUGTRACE
/*
unsigned char GetATTR(unsigned char index)
{
	unsigned char i = inb(0x3da);
  	outb(0x3c0, index);
	return inb(0x3c1);
}
unsigned char GetGRX(unsigned char index)
{
  	outb(0x3ce, index);
	return inb(0x3cf);
}
*/

void DumpMXICRegs(void)
{
	int i;
	ErrorF(" : CRTC (");
	for (i=0; i<=0x3F; i++)
		ErrorF("%02x,",GetCRTC(i));
	ErrorF(")\n");
	ErrorF(" : SEQ (");
	for (i=0; i<=0x1F; i++)
		ErrorF("%02x,",GetSEQ(i));
	ErrorF(")\n");
/*
	ErrorF(" : ATTR (");
	for (i=0; i<=0x14; i++)
		ErrorF("%02x,",GetATTR(i));
	ErrorF(")\n");
	ErrorF(" : GRX (");
	for (i=0; i<=0x8; i++)
		ErrorF("%02x,",GetGRX(i));
	ErrorF(")\n");
	ErrorF(" : MISC (%02x)\n",inb(0x3cc));
*/
}
#endif

/*
 * MXICClockSelect --
 * 	select one of the possible clocks ...
 */
static Bool
MXICClockSelect(no)
	int no;
{
	static unsigned char save1, save2, save3;
	static unsigned char save4, save5, save6, save7;
	unsigned char temp;
	unsigned tempw;

	/*
	 * CS0 and CS1 are in MiscOutReg
	 *
	 * For 8900B, 8900C, 8900CL and 9000, CS2 is bit 0 of
	 * New Mode Control Register 2.
	 *
	 * For 8900CL, CS3 is bit 4 of Old Mode Control Register 1.
	 *
	 * For 9000, CS3 is bit 6 of New Mode Control Register 2.
	 */
	switch(no)
	{
	case CLK_REG_SAVE:
		/*
		   save standard VGA clock selection
		 */
		save1 = inb(0x3CC);

		/*
		  save DAC mode
		*/
		outb(0x3C4, 0x11);
		save7 = inb(0x3c5);

		/* 
		   save N, M value for VCG DCLK 
		 */
		outb(0x3C4, 0x18);
		save2 = inb(0x3c5);
		outb(0x3C4, 0x19);
		save3 = inb(0x3c5);

		/*
		   save VCG clock source selection
		 */
		outb(0x3C4, 0x1d);
		save4 = inb(0x3c5);

		/*
		   save system control register
		 */
		outb(vgaCRIndex, SystemControlRegIdx);
		save5 = inb(vgaCRReg);

		/*
		   save bank register
		 */
		outb(vgaCRIndex, BankRegIdx);
		save6 = inb(vgaCRReg);

#ifdef DEBUGTRACE
		ErrorF("MXICClockSelect: CLK_REG_SAVE\n");
		DumpMXICRegs();
#endif
		break;

	case CLK_REG_RESTORE:
		/* 
		   sync reset by programming 1 to sequencer register 3C4h
		   index 0, this is required becasue programming to
		   3C4h/1Dh will be effect only in sync reset 
		 */
		outw(0x3c4, 0x0100);

		/*
		  restore DAC mode
		*/
		outb(0x3C4, 0x11);
		outb(0x3C5, save7);

		/*
		   restore standard VGA clock selection
		 */
		outb(0x3C2, save1);

		/*
		   restore VCG clock source selection
		 */
		outb(0x3C4, 0x1d);
		outb(0x3C5, save4);

		/* 
		   restore N, M value for VCG DCLK 
		 */
		outb(0x3C4, 0x18);
		outb(0x3C5, save2);
		outb(0x3C4, 0x19);
		outb(0x3C5, save3);
		outb(0x3c4, 0x1c);	/* trigger to load */
		outb(0x3c5, 0x10);
		outb(0x3c5, 0x00);

		/*
		   restore system control register
		 */
		outb(vgaCRIndex, SystemControlRegIdx);
		outb(vgaCRReg, save5);

		/*
		   restore bank register
		 */
		outb(vgaCRIndex, BankRegIdx);
		outb(vgaCRReg, save6);

		/* 
		   start sequencer by programming 3 to sequencer register 
		   3C4h index 0 
		 */
		outw(0x3c4, 0x0300);

#ifdef DEBUGTRACE
		ErrorF("MXICClockSelect: CLK_REG_RESTORE\n");
		DumpMXICRegs();
#endif
		break;

	default:
		/* 
		   Programming to 3CCh is invalid here because I will use
		   extended VCG to select clock.  These two lines are 
		   the standard VGA way that sample code does. 
		 */
      		temp = inb(0x3CC);
      		outb(0x3C2, ( temp & 0xf3) | ((no << 2) & 0x0C));

		/* 
		   sync reset by programming 1 to sequencer register 3C4h
		   index 0, this is required becasue programming to
		   3C4h/1Dh will be effect only in sync reset 
		 */
		outw(0x3c4, 0x0100);

		/* 
		   Set to extended mode to ensure correct clock probing 
		 */
		tempw = 0x7700 | SystemControlRegIdx;
		outw(vgaCRIndex, tempw);

		/* 
	 	   set bit 5 of 3C4h/1Dh to select VCG clock source
		   from extended(internal) VCG instead of standard 
		   register 3C2h 
		 */
		outw(0x3c4, 0x2f1d);

		/* 
		   load N, M value for VCG DCLK 
		 */
		outb(0x3c4, 0x18);
		outb(0x3c5, dclk[no & 0x0F].SEQ18);
		outb(0x3c4, 0x19);
		outb(0x3c5, dclk[no & 0x0F].SEQ19);
		outb(0x3c4, 0x1c);	/* trigger to load */
		outb(0x3c5, 0x10);
		outb(0x3c5, 0x00);

		/* 
		   start sequencer by programming 3 to sequencer register 
		   3C4h index 0 
		 */
		outw(0x3c4, 0x0300);

#ifdef DEBUGTRACE
		ErrorF("MXICClockSelect: #%i (%02X,%02X)\n", no, 
			(int)dclk[no & 0x0F].SEQ18,(int)dclk[no & 0x0F].SEQ19);
		DumpMXICRegs();
#endif
	}
	return(TRUE);
}

/* 
 * MXICProbe --
 * 	check up whether a Macronix based board is installed
 */
static Bool
MXICProbe()
{
	MXICPCIInformation *pciInfo = NULL;
  	int numClocks;
  	unsigned char temp;

#ifdef DEBUGTRACE
	ErrorF("\n%s MXICProbe ", XCONFIG_PROBED);
#endif

	/*
         * Set up I/O ports to be used by this card
	 */
	xf86ClearIOPortList(vga256InfoRec.scrnIndex);
	xf86AddIOPorts(vga256InfoRec.scrnIndex, Num_VGA_IOPorts, VGA_IOPorts);

   	if (vga256InfoRec.chipset)
    	{
		/*
		 * If chipset from XF86Config matches...
		 */
		ErrorF("%s Test for chipset string '%s'\n", XCONFIG_PROBED, vga256InfoRec.chipset);
		if (!StrCaseCmp(vga256InfoRec.chipset, "mxic"))
		{
			ErrorF("  - mxic is no longer valid.  Use one of\n");
			ErrorF("    the names listed by the -showconfig option\n");
			return(FALSE);
		}
		if (!StrCaseCmp(vga256InfoRec.chipset, MXICIdent(0)))
		{
			ErrorF("  - StrCaseCmp Match MX86250\n");
			MXICchipset = MXIC_MX86250;
			MXIC.ChipSetRead      = MXICSetRead86250;
			MXIC.ChipSetWrite     = MXICSetWrite86250;
			MXIC.ChipSetReadWrite = MXICSetReadWrite86250;
			SystemControlRegIdx = 0x25;
			BankRegIdx          = 0x2e;
		}
		else if (!StrCaseCmp(vga256InfoRec.chipset, MXICIdent(1)))
		{
			ErrorF("  - StrCaseCmp Match MX86251\n");
			MXICchipset = MXIC_MX86251;
			MXIC.ChipSetRead      = MXICSetRead86251;
			MXIC.ChipSetWrite     = MXICSetWrite86251;
			MXIC.ChipSetReadWrite = MXICSetReadWrite86251;
			SystemControlRegIdx = 0x32;
			BankRegIdx          = 0x34;
		}
		else
		{
			ErrorF("  - Not a MXIC SVGA Chipset!\n");
			return(FALSE);
		}
		MXICEnterLeave(ENTER);
    	}
	else
    	{
   		/* Start with PCI probing, this should get us quite far already.
   		 * For now, we only use the PCI probing. 
		 */

	   	pciInfo = mxicGetPCIInfo();
   		if (!pciInfo)
		{
			ErrorF("  - Cannot get PCI information!\n");
      			return FALSE;
		}

	   	if (pciInfo && pciInfo->MemBase)
      			vga256InfoRec.MemBase = pciInfo->MemBase;

	   	if (pciInfo)
		{
      			if(pciInfo->ChipType != MXIC_MX86250 && 
	         	   pciInfo->ChipType != MXIC_MX86251)
			{
             			ErrorF("%s %s: Unsupported (non-MXIC) MXIC chipset detected!\n", 
                		        XCONFIG_PROBED, vga256InfoRec.name);
#ifndef DEBUGTRACE
          			if (xf86Verbose > 1)
#endif
             				ErrorF("%s %s: Unsupported (non-MXIC) MXIC chipset detected!\n", 
                				XCONFIG_PROBED, vga256InfoRec.name);

          			return FALSE;
	          	}
      			else 
			{
				mxicPriv.MXIC_MMIO_REGBASE = MX8625x_MMIO_REGBASE;
				mxicPriv.MXIC_MMIO_REGSIZE = MX8625x_MMIO_REGSIZE;
         			mxicPriv.chip = pciInfo->ChipType;

				switch (mxicPriv.chip)
				{
				case MXIC_MX86250:
				  MXICchipset = MXIC_MX86250;
				  MXIC.ChipSetRead      = MXICSetRead86250;
				  MXIC.ChipSetWrite     = MXICSetWrite86250;
				  MXIC.ChipSetReadWrite = MXICSetReadWrite86250;
				  SystemControlRegIdx = 0x25;
				  BankRegIdx          = 0x2e;
				  break;
				case MXIC_MX86251:
				  MXICchipset = MXIC_MX86251;
				  MXIC.ChipSetRead      = MXICSetRead86251;
				  MXIC.ChipSetWrite     = MXICSetWrite86251;
				  MXIC.ChipSetReadWrite = MXICSetReadWrite86251;
				  SystemControlRegIdx = 0x32;
				  BankRegIdx          = 0x34;
				  break;
				}

         			ErrorF("%s %s: Detected MXIC %s\n",XCONFIG_PROBED,
            				vga256InfoRec.name, xf86TokenToString(mxicChipTable, mxicPriv.chip));
         			ErrorF("%s %s: using driver for chipset \"%s\"\n",XCONFIG_PROBED, 
            				vga256InfoRec.name, MXICIdent(mxicPriv.chip));
	 		}

			MXICEnterLeave(ENTER);

		}
		else {
		  /* Old style register based checking */
		
		  unsigned short DeviceIDVal;
		  char *MXICName;
		  char *TreatAs = NULL;
	
		  MXICEnterLeave(ENTER);

		  /* 
		   * Check first that we have a Macronix card.
		   */
		  outb(vgaCRIndex, 0xF2);
		  temp = inb(vgaCRReg);
		  DeviceIDVal = (unsigned short)temp << 8;
		  outb(vgaCRIndex, 0xF1);
		  temp = inb(vgaCRReg) & 0xf0;
		  DeviceIDVal += temp;

		  /* 
		   * Is it a Macronix card ?? 
		   */
		  if (DeviceIDVal != 0x8620)
		  {
		      /*
		       * Nope, so quit
		       */
		      ErrorF("\nNot a MXIC SVGA card!\n");
		      MXICEnterLeave(LEAVE);
		      return(FALSE);
		  }

		  /* 
		   * We've found a Macronix card, now check the version.
		   */
		  MXICchipset = -1;
		  outb(vgaCRIndex, 0xF1);
		  temp = inb(vgaCRReg);
		  switch (temp)
		  {
		    case 0x25:
		      MXICchipset = MXIC_MX86250;
		      MXICName = "MX86250";
		      MXIC.ChipSetRead      = MXICSetRead86250;
		      MXIC.ChipSetWrite     = MXICSetWrite86250;
		      MXIC.ChipSetReadWrite = MXICSetReadWrite86250;
		      SystemControlRegIdx = 0x25;
		      BankRegIdx          = 0x2e;
		      outb(vgaCRIndex, 0x28);
		      vga256InfoRec.MemBase = ((int32)inb(vgaCRReg))<<25;
		      break;
		    case 0x26:
		      MXICchipset = MXIC_MX86251;
		      MXICName = "MX86251";
		      MXIC.ChipSetRead      = MXICSetRead86251;
		      MXIC.ChipSetWrite     = MXICSetWrite86251;
		      MXIC.ChipSetReadWrite = MXICSetReadWrite86251;
		      SystemControlRegIdx = 0x32;
		      BankRegIdx          = 0x34;
		      outb(vgaCRIndex, 0x38);
		      vga256InfoRec.MemBase = ((int32)inb(vgaCRReg))<<25;
		      break;
		    default:
		      MXICName = "UNKNOWN";
		  }
		  ErrorF("%s Macronix chipset version: 0x%02x (%s)\n", 
			 XCONFIG_PROBED, temp, MXICName);
		  if (TreatAs != (char *)NULL)
		  {
		      ErrorF("%s \tDriver will treat chipset as: %s\n",
			     XCONFIG_PROBED, TreatAs);
		  }
		  if (MXICchipset == -1)
		  {
		      ErrorF("Unknown Macronix chipset.\n");
		      MXICEnterLeave(LEAVE);
		      return(FALSE);
		  }
		}
    	}
 
 	/* 
	 * How much Video Ram have we got?
	 */
    	if (!vga256InfoRec.videoRam)
    	{
		unsigned char temp;

		outb(vgaCRIndex, 0x57); 
		temp = inb(vgaCRReg);

		switch ((temp >> 3) & 0x03)
		{
		case 0: 
			vga256InfoRec.videoRam = 1024; 
			break;
		case 1: 
			vga256InfoRec.videoRam = 2048; 
			break;
		case 2: 
			vga256InfoRec.videoRam = 4096; 
			break;
		}
     	}

	/*
	 * If clocks are not specified in XF86Config file, probe for them
	 */
    	if (!vga256InfoRec.clocks) 
	{
		numClocks = 16;
		vgaGetClocks(numClocks, MXICClockSelect); 
	}
	
   	/* And map MMIO memory, abort if we cannot */

	mxicMmioMem = xf86MapVidMem(vga256InfoRec.scrnIndex, MMIO_REGION, 
      		(pointer)(vga256InfoRec.MemBase + mxicPriv.MXIC_MMIO_REGBASE), mxicPriv.MXIC_MMIO_REGSIZE);

	if(mxicMmioMem == NULL) 
      		FatalError("MXIC: Cannot map MMIO registers!\n");
   
	/*
	 * Get to New Mode.
	 */

	vga256InfoRec.chipset = MXICIdent(MXICchipset);
	/* vga256InfoRec.bankedMono = TRUE; */
	vga256InfoRec.bankedMono = FALSE;

#ifdef XFreeXDGA
   	vga256InfoRec.directMode = XF86DGADirectPresent;
#endif

	/* Initialize option flags allowed for this driver */
	OFLG_SET(OPTION_16CLKS, &MXIC.ChipOptionFlags);
        /* OFLG_SET(OPTION_NOACCEL, &MXIC.ChipOptionFlags); */
        OFLG_SET(OPTION_HW_CURSOR, &MXIC.ChipOptionFlags);
	OFLG_SET(OPTION_LINEAR, &MXIC.ChipOptionFlags);

   	mxicPriv.NoPCIRetry = 1;
   	MXIC.ChipLinearBase = vga256InfoRec.MemBase;
   	MXIC.ChipLinearSize = vga256InfoRec.videoRam * 1024;

    	return(TRUE);
}

/*
 * MXICEnterLeave --
 * 	enable/disable io-mapping
 */
static void
MXICEnterLeave(enter)
	Bool enter;
{
  	unsigned char temp;

#ifdef XFreeXDGA
	if (vga256InfoRec.directMode&XF86DGADirectGraphics && !enter)
		return;
#endif
  	if (enter)
    	{
      		xf86EnableIOPorts(vga256InfoRec.scrnIndex);
		vgaIOBase = (inb(0x3CC) & 0x01) ? 0x3D0 : 0x3B0;
                vgaCRIndex = vgaIOBase + 4;
                vgaCRReg = vgaIOBase + 5;
      		outb(vgaCRIndex, 0x11); temp = inb(vgaCRReg);
      		outb(vgaCRReg, temp & 0x7F);
		outw(vgaCRIndex, 0x8819);
#ifdef DEBUGTRACE
		ErrorF("\n%s MXICEnterLeave(TRUE)\n", XCONFIG_PROBED);
#endif
    	}
  	else
    	{
		outw(vgaCRIndex, 0x0019);
      		xf86DisableIOPorts(vga256InfoRec.scrnIndex);
#ifdef DEBUGTRACE
		ErrorF("\n%s MXICEnterLeave(FALSE)\n", XCONFIG_PROBED);
#endif
    	}
}

/*
 * MXICRestore --
 *      restore a video mode
 */
static void
MXICRestore(restore)
     	vgaMXICPtr restore;
{
#ifdef DEBUGTRACE
	ErrorF("\nMXICRestore\n  before");
	DumpMXICRegs();
#endif

#ifdef FORCEMODEREG
	if (vga256InfoRec.bitsPerPixel != 8)
	{
		int i;
		for (i=0; i<0x3f; i++)
		{
			outb(vgaCRIndex, i);
			outb(vgaCRReg, restore->CRTCALL[i]);
		}
		for (i=0; i<0x1f; i++)
		{
			outb(0x3C4, i);
			outb(0x3C5, restore->SEQALL[i]);
		}
	}
#endif

	/* Screen off */
	/* outw(0x3c4, 0x2101); */

	/* 
	   sync reset by programming 1 to sequencer register 3C4h
	   index 0, this is required becasue programming to
	   3C4h/1Dh will be effect only in sync reset 
	 */
	outw(0x3c4, 0x0100);

	/*
	 * Restore vendor specific extended registers
	 */

	outb(vgaCRIndex, 0x21);
	outb(vgaCRReg, restore->CRTC21);
	outb(vgaCRIndex, 0x22);
	outb(vgaCRReg, restore->CRTC22);
	outb(vgaCRIndex, 0x23);
	outb(vgaCRReg, restore->CRTC23);
	outb(vgaCRIndex, 0x24);
	outb(vgaCRReg, restore->CRTC24);
	outb(vgaCRIndex, 0x66);
	outb(vgaCRReg, restore->CRTC66);
	outb(vgaCRIndex, SystemControlRegIdx);
	outb(vgaCRReg, restore->SystemControl);
	outb(0x3c4, 0x1d);
	outb(0x3c5, restore->SEQ1D);
	outb(0x3c4, 0x11);
	outb(0x3c5, restore->SEQ11);
	outb(0x3c4, 0x18);
	outb(0x3c5, restore->SEQ18);
	outb(0x3c4, 0x19);
	outb(0x3c5, restore->SEQ19);
	/* trigger to load */
	outb(0x3c4, 0x1c);
	outb(0x3c5, 0x10);
	outb(0x3c5, 0x00);

	if (MXICchipset == MXIC_MX86250)
	{
		outb(vgaCRIndex, 0x33);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC33);
		outb(vgaCRIndex, 0x34);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC34);
		outb(vgaCRIndex, 0x35);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC35);
		outb(vgaCRIndex, 0x36);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC36);
		outb(vgaCRIndex, 0x37);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC37);
		outb(vgaCRIndex, 0x38);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC38);
		outb(vgaCRIndex, 0x39);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC39);
		outb(vgaCRIndex, 0x3a);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC3A);
		outb(vgaCRIndex, 0x3b);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC3B);
		outb(vgaCRIndex, 0x3c);
		outb(vgaCRReg, restore->extreg.MX86250.CRTC3C);
	}
	else if (MXICchipset == MXIC_MX86251)
	{
		outb(vgaCRIndex, 0x1f);
		outb(vgaCRReg, restore->extreg.MX86251.CRTC1F);
		outb(vgaCRIndex, 0x20);
		outb(vgaCRReg, restore->extreg.MX86251.CRTC20);
	}

	/* 
	   start sequencer by programming 3 to sequencer register 
	   3C4h index 0 
	 */
	outw(0x3c4, 0x0300);

	/* Screen on */
	/* outw(0x3c4, 0x0101); */

	/*
	 * Now restore generic VGA Registers
	 */
	vgaHWRestore((vgaHWPtr)restore);

#ifdef DEBUGTRACE
	ErrorF("   after");
	DumpMXICRegs();
#endif

#ifdef FORCEMODEREG
	if (restore == new)
	{
		MXICForceRestore(extmode);
		extmode = NULL;
	}
#ifdef DEBUGTRACE
	ErrorF("   after");
	DumpMXICRegs();
#endif
#endif
}

/*
 * MXICSave --
 *      save the current video mode
 */
static void *
MXICSave(save)
     	vgaMXICPtr save;
{
	int i;

#ifdef DEBUGTRACE
	ErrorF("\nMXICSave\n");
	DumpMXICRegs();
#endif
	/*
	 * Save generic VGA registers
	 */
  	save = (vgaMXICPtr)vgaHWSave((vgaHWPtr)save, sizeof(vgaMXICRec));

	/*
	 * Save vendor specific extended registers
	 */
	outb(vgaCRIndex, 0x21);
	save->CRTC21 = inb(vgaCRReg);
	outb(vgaCRIndex, 0x22);
	save->CRTC22 = inb(vgaCRReg);
	outb(vgaCRIndex, 0x23);
	save->CRTC23 = inb(vgaCRReg);
	outb(vgaCRIndex, 0x24);
	save->CRTC24 = inb(vgaCRReg);
	outb(vgaCRIndex, 0x66);
	save->CRTC66 = inb(vgaCRReg);
	outb(vgaCRIndex, SystemControlRegIdx);
	save->SystemControl = inb(vgaCRReg);
	outb(0x3c4, 0x11);
	save->SEQ11 = inb(0x3c5);
	outb(0x3c4, 0x18);
	save->SEQ18 = inb(0x3c5);
	outb(0x3c4, 0x19);
	save->SEQ19 = inb(0x3c5);
	outb(0x3c4, 0x1d);
	save->SEQ1D = inb(0x3c5);

	if (MXICchipset == MXIC_MX86250)
	{
		outb(vgaCRIndex, 0x33);
		save->extreg.MX86250.CRTC33 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x34);
		save->extreg.MX86250.CRTC34 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x35);
		save->extreg.MX86250.CRTC35 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x36);
		save->extreg.MX86250.CRTC36 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x37);
		save->extreg.MX86250.CRTC37 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x38);
		save->extreg.MX86250.CRTC38 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x39);
		save->extreg.MX86250.CRTC39 = inb(vgaCRReg);
		outb(vgaCRIndex, 0x3a);
		save->extreg.MX86250.CRTC3A = inb(vgaCRReg);
		outb(vgaCRIndex, 0x3b);
		save->extreg.MX86250.CRTC3B = inb(vgaCRReg);
		outb(vgaCRIndex, 0x3c);
		save->extreg.MX86250.CRTC3C = inb(vgaCRReg);
	}
	else if (MXICchipset == MXIC_MX86251)
	{
		outb(vgaCRIndex, 0x1f);
		save->extreg.MX86251.CRTC1F = inb(vgaCRReg);
		outb(vgaCRIndex, 0x20);
		save->extreg.MX86251.CRTC20 = inb(vgaCRReg);
	}

#ifdef FORCEMODEREG
	if (vga256InfoRec.bitsPerPixel != 8)
	{
		for (i=0; i<0x3f; i++)
		{
			outb(vgaCRIndex, i);
			save->CRTCALL[i] = inb(vgaCRReg);
		}
		for (i=0; i<0x1f; i++)
		{
			outb(0x3C4, i);
			save->SEQALL[i] = inb(0x3C5);
		}
	}
#endif

  	return ((void *) save);
}

/*
 * MXICInit --
 *      Handle the initialization, etc. of a screen.
 */
static Bool
MXICInit(mode)
    	DisplayModePtr mode;
{
	unsigned tmp, tmptot, tmpss, tmpse, tmpbs, tmpbe;
	unsigned widthbytes;

#ifdef DEBUGTRACE
	ErrorF("\nMXICInit: %ix%i(Clock=%i,SynthClock=%x)\n",
		mode->CrtcHDisplay,mode->CrtcVDisplay,
		mode->Clock,mode->SynthClock);
#endif

	/*
	 * Initialize generic VGA Registers
	 */
#ifdef DEBUGTRACE
 	ErrorF("Initialize generic VGA Registers\n");
#endif

	vgaHWInit(mode, sizeof(vgaMXICRec)); 

	/*
	 * Now do Macronix-specific stuff.
	 */
	widthbytes = (vga256InfoRec.virtualX * (vgaBitsPerPixel / 8)) >> 3;
	new->std.CRTC[19] = widthbytes & 0xFF;

	new->std.CRTC[20] = 0x40;
	new->std.CRTC[23] = 0xA3;

	if (vga256InfoRec.bitsPerPixel != 8)
	{
		/* original Attribute value from standard VGA init
		   (0x10,0x11,0x12,0x13,0x14)=(0x41,0xff,0x0f,0x00,0x00) */
		new->std.Attribute[0x10] = 0x01;
		new->std.Attribute[0x11] = 0x01;
		new->std.Attribute[0x12] = 0x0f;
		new->std.Attribute[0x13] = 0x13;
		new->std.Attribute[0x14] = 0x00;
	}

#ifdef DEBUGTRACE
	ErrorF("Initialize extended VGA Registers\n");
#endif
	new->CRTC21 = 0x00;
	tmp    = (mode->CrtcHDisplay >> 3) - 1;
	tmptot = (mode->CrtcHTotal >> 3) - 5;
	tmpss  = (mode->CrtcHSyncStart >> 3);
	tmpse  = (mode->CrtcHSyncEnd >> 3);
	tmpbs  = (mode->CrtcHSyncStart >> 3) - 1;
	tmpbe  = (mode->CrtcHSyncEnd >> 3);
	new->CRTC22 = 	((tmptot & 0x100)>>8) | 
		      	((tmp & 0x100)>>7) |
			((tmpss & 0x100)>>6) |
			((tmpbs & 0x100)>>5) |
			((tmpse & 0x20)>>1) |
			((tmpbe & 0x40));
	tmp    = mode->CrtcVDisplay - 1;
	tmptot = mode->CrtcVTotal - 2;
	tmpss  = mode->CrtcVSyncStart;
	tmpse  = mode->CrtcVSyncEnd;
	tmpbs  = mode->CrtcVSyncStart - 1;
	tmpbe  = widthbytes; 	/* borrow for offset */
	new->CRTC23 = 	((tmptot & 0x400)>>10) | 
		      	((tmp & 0x400)>>9) |
			((tmpss & 0x400)>>8) |
			((tmpbs & 0x400)>>7) |
			((tmpse & 0x10)<<1) |
			((tmpbe & 0x300)>>2);
	/* new->CRTC24 = 0x60; */
	new->CRTC24 = 0;
	/*
	switch (mode->CrtcHDisplay)
	{
		case 640:
			new->SEQ1D = 0x2f; 
	    		new->CRTC24 = 0x60;
			break;
		case 800:
			new->SEQ1D = 0x64; 
	    		new->CRTC24 = 0x60;
			break;
		case 1024:
			new->SEQ1D = 0x22; 
	    		new->CRTC24 = 0x20;
			break;
		case 1280:
			new->SEQ1D = 0x2f; 
	    		new->CRTC24 = 0x20;
			break;
		case 1600:
			new->SEQ1D = 0x6a; 
	    		new->CRTC24 = 0x20;
			break;
		default:
	    		new->CRTC24 = 0;
			break;
	}
	*/
	if (mode->Flags & V_INTERLACE)
		new->CRTC24 |= 0x8;
	switch (vga256InfoRec.bitsPerPixel)
	{
		case 8: 
#ifdef DEBUGTRACE
			ErrorF("  - select 8 bpp\n");
#endif
			if (mode->CrtcHDisplay == 1600)
				new->CRTC24 |= 0x43;
			else
				new->CRTC24 |= 0x3;
			new->SEQ11 = 0;
			break;
		case 15: 
#ifdef DEBUGTRACE
			ErrorF("  - select 15 bpp\n");
#endif
			new->CRTC24 |= 0x7;
			new->SEQ11 = 0x20;
			break;
		case 16: 
#ifdef DEBUGTRACE
			ErrorF("  - select 16 bpp\n");
#endif
			new->CRTC24 |= 0x4;
			new->SEQ11 = 0x40;
			break;
		case 24: 
#ifdef DEBUGTRACE
			ErrorF("  - select 24 bpp\n");
#endif
			new->CRTC24 |= 0x5;
			new->SEQ11 = 0x60;
			break;
		case 32: 
#ifdef DEBUGTRACE
			ErrorF("  - select 32 bpp\n");
#endif
			new->CRTC24 |= 0x6;
			new->SEQ11 = 0x60;
			break;
		default: 
#ifdef DEBUGTRACE
			ErrorF("  - default 8 bpp\n");
#endif
			new->CRTC24 |= 0x3;
			new->SEQ11 = 0;
			break;
	}
	new->SystemControl = 0x77;
	new->SEQ18 = dclk[mode->Clock & 0x0F].SEQ18;
	new->SEQ19 = dclk[mode->Clock & 0x0F].SEQ19;
	new->SEQ1D = 0x2f; 
	if (MXICchipset == MXIC_MX86251)
	{
	        widthbytes = (mode->CrtcHDisplay * (vgaBitsPerPixel / 8)) >> 3;
		new->extreg.MX86251.CRTC1F = widthbytes & 0xFF;
		new->extreg.MX86251.CRTC20 = (widthbytes >> 8) | 0x50; 
	}
	else if (MXICchipset == MXIC_MX86250)
	{
		new->extreg.MX86250.CRTC33 = 0x1f;
		new->extreg.MX86250.CRTC34 = 0xf7;
		new->extreg.MX86250.CRTC35 = 0x77;
		new->extreg.MX86250.CRTC36 = 0x1e;
		new->extreg.MX86250.CRTC37 = 0x01;
		new->extreg.MX86250.CRTC38 = 0x1e;
		new->extreg.MX86250.CRTC39 = 0x5d;
		new->extreg.MX86250.CRTC3A = 0x00;

		tmp = (vga256InfoRec.bitsPerPixel + 7) >> 3;

		if (vga256InfoRec.videoRam == 1024)
		{
			tmp = (mode->CrtcHDisplay * tmp) >> 2;
			new->extreg.MX86250.CRTC3B = tmp & 0x00ff;
			new->extreg.MX86250.CRTC3C = tmp & 0x0300;
		}
	 	else	
		{
			tmp = (mode->CrtcHDisplay * tmp) >> 3;
			new->extreg.MX86250.CRTC3B = tmp & 0x00ff;
			new->extreg.MX86250.CRTC3C = tmp & 0x0300;
		}
	}

#ifdef FORCEMODEREG
	if (vga256InfoRec.bitsPerPixel != 8)
	{
		int i;
		for (i=0; i<0x3f; i++)
		{
			outb(vgaCRIndex, i);
			new->CRTCALL[i] = inb(vgaCRReg);
		}
		for (i=0; i<0x1f; i++)
		{
			outb(0x3C4, i);
			new->SEQALL[i] = inb(0x3C5);
		}
		extmode = mode;
	}
#endif
        return(TRUE);
}

#ifdef FORCEMODEREG
static Bool
MXICForceRestore(mode)
    	DisplayModePtr mode;
{
	int residx = 0;
	int coloridx = 0;
	USHORT *modetbl = NULL;
	int modetblidx = 0;

	ErrorF("MXICForceRestore\n");
	if (vga256InfoRec.bitsPerPixel != 16)
	{
		ErrorF("MXICForceRestore: not 16bpp mode!\n");
		return FALSE;
	}

		new->CRTC24 |= 0x8;
	switch (vga256InfoRec.bitsPerPixel)
	{
		case 16: 
			coloridx = COLOR_16BPP;
			break;
	}
	switch (mode->CrtcHDisplay)
	{
		case 640:
			residx = RES_640x480;
			break;
		case 800:
			residx = RES_800x600;
			break;
		case 1024:
			residx = RES_1024x768;
			break;
		case 1152:
			residx = RES_1152x864;
			break;
		case 1280:
			residx = RES_1280x1024;
			break;
		case 1600:
			residx = RES_1600x1200;
			break;
	}

	if (MXICchipset == MXIC_MX86251)
	{
		modetbl = MX_86251_MODES[coloridx][residx];
	}
	else if (MXICchipset == MXIC_MX86250)
	{
		modetbl = MX_86250_MODES[coloridx][residx];
	}

	if (modetbl == NULL)
	{
		ErrorF("MXICForceRestore: null mode table\n");
		return FALSE;
	}

	while (modetbl[modetblidx] != EOD)
	{
		int curreg = 0;
		int count = 0;
		int i;
		unsigned char tmpregval;
		switch (modetbl[modetblidx++])
		{
			case OB:
				ErrorF("OB(%x,%x)\n",modetbl[modetblidx],modetbl[modetblidx+1]);
				outb(modetbl[modetblidx], modetbl[modetblidx+1]);
				modetblidx += 2;
				break;
			case OW:
				ErrorF("OW(%x,%x)\n",modetbl[modetblidx],modetbl[modetblidx+1]);
				outw(modetbl[modetblidx], modetbl[modetblidx+1]);
				modetblidx += 2;
				break;
			case OWM:
				curreg = modetbl[modetblidx++];
				count = modetbl[modetblidx++];
				ErrorF("OWM(%x,%x):",curreg,count);
				while (count--)
				{
					ErrorF("%x,",modetbl[modetblidx]);
					outw(curreg, modetbl[modetblidx++]);
				}
				ErrorF("\n");
				break;
			case METAOUT+MASKOUT:
				curreg = modetbl[modetblidx++];
				tmpregval = inb(curreg);
				tmpregval &= modetbl[modetblidx++];
				outb(curreg, tmpregval);
				tmpregval |= modetbl[modetblidx++];
				outb(curreg, tmpregval);
				ErrorF("METAOUT+MASKOUT(%x,%x)\n",curreg,tmpregval);
				break;
			case METAOUT+INDXOUT:
				curreg = modetbl[modetblidx++];
				count = modetbl[modetblidx++];
				i = modetbl[modetblidx++];
				while (count--)
				{
					ErrorF("METAOUT+INDXOUT(%x,%x)=%x\n",curreg,i,modetbl[modetblidx]);
					outb(curreg, i);
					outb(curreg+1, modetbl[modetblidx++]);
					i++;
				}
				break;
			case IB:
				ErrorF("IB(%x)\n",modetbl[modetblidx]);
				tmpregval = inb(modetbl[modetblidx++]);
				break;
		}
	}

	return TRUE;
}
#endif

/*
 * MXICAdjust --
 *      adjust the current video frame to display the mousecursor
 */

static void 
MXICAdjust(x, y)
	int x, y;
{
	int base;

#ifdef DEBUGTRACE
	ErrorF("\nMXICAdjust(%i,%i) [(%i)/(%i,%i)]\n",x,y,
		vga256InfoRec.displayWidth, 
		vga256InfoRec.virtualX, vga256InfoRec.virtualY);
#endif
	
	if (vgaBitsPerPixel == 24)
		/* base = ((y * vga256InfoRec.displayWidth + ((x+3)&~3)) * ((vgaBitsPerPixel+7)/8)) >> 2; */
		base = ((y * vga256InfoRec.displayWidth + (x&~3)) * ((vgaBitsPerPixel+7)/8)) >> 2;
	else
		base = ((y * vga256InfoRec.displayWidth + x) * ((vgaBitsPerPixel+7)/8)) >> 2;

	outw(vgaCRIndex, (base & 0x00ff00) | 0x0c);
	outw(vgaCRIndex, ((base & 0x00ff)<<8) | 0x0d);
	outw(vgaCRIndex, ((base & 0xff0000)>>8) | 0x21);

	return;
}

/*
 * MXICValidMode --
 *
 */
static Bool
MXICValidMode(mode)
DisplayModePtr mode;
{
#ifdef DEBUGTRACE
	ErrorF("\nMXICValidMode: %ix%i\n",mode->CrtcHDisplay,mode->CrtcVDisplay);
#endif
	switch (mode->CrtcHDisplay)
	{
		case 640:
		case 800:
		case 1024:
		case 1152:
		case 1280:
		case 1600:
			return MODE_OK;
	}
	return MODE_BAD;
}


/* This function inits the frame buffer. Right now, it is is rather limited 
 * but the hardware cursor hooks should probably end up here 
 */

void 
MXICFbInit()
{

   /* Call MXICAccelInit to setup the XAA accelerated functions */
   if(!OFLG_ISSET(OPTION_NOACCEL, &vga256InfoRec.options))
      if(vgaBitsPerPixel == 24) 
         MXICAccelInit24();
      else 
         MXICAccelInit();

   if(OFLG_ISSET(OPTION_PCI_RETRY, &vga256InfoRec.options))
      if(OFLG_ISSET(OPTION_PCI_BURST_ON, &vga256InfoRec.options)) {
         mxicPriv.NoPCIRetry = 0;
         }
      else {
         mxicPriv.NoPCIRetry = 1;   
         ErrorF("%s %s: \"pci_retry\" option requires \"pci_burst\".\n",
              XCONFIG_GIVEN, vga256InfoRec.name);
         }

   if (!OFLG_ISSET(OPTION_SW_CURSOR, &vga256InfoRec.options)) {
      vgaHWCursor.Initialized = TRUE;
      vgaHWCursor.Init = MXICCursorInit;
      vgaHWCursor.Restore = MXICRestoreCursor;
      vgaHWCursor.Warp = MXICWarpCursor;
      vgaHWCursor.QueryBestSize = MXICQueryBestSize;
#ifndef DEBUGTRACE
      if (xf86Verbose)
#endif
                ErrorF("%s %s: %s: Using hardware cursor\n",
                        XCONFIG_PROBED, vga256InfoRec.name,
                        vga256InfoRec.chipset);
    }
#ifdef DEBUGTRACE
    else
                ErrorF("%s %s: %s: Using software cursor\n",
                        XCONFIG_PROBED, vga256InfoRec.name,
                        vga256InfoRec.chipset);
#endif

}

